// (C)1997 Christian Janoff
// ZIPCODE.CC

#include <iostream.h>
#include <iomanip.h>
#include "c1541.h"
#include "file.h"
#include "error.h"
#include "pkstring.h"

void c1541disk::open_zipcode_file(char *name, char number, int mode)
{
  pkstring fname,
           base=name,
           path=name;

  base.conv2basename();
  path.conv2pathname();

  if (mode==READ) fname=path;
  if (fname.string()) fname=fname+"/";

  if (base[1]=='!' && base[0]>'0' && base[0]<'5')
    {
      base[0]=number;
      fname=fname+base;
    }
  else
    fname=fname+number+"!"+base;
  iofile_open(fname.string(),mode);
}

void c1541disk::close_zipcode_file(void)
{
 iofile_close();
}

int c1541disk::zipcode(pkstring filename, int mode)
{
  int t,s;
  int nbzipsectors=0;
  int nbbytes=0;
  int err;

  if (mode==WRITE && filename.has_suffix(".d64")) filename.remove_suffix(".d64");

  open_zipcode_file(filename.string(),'1',mode);

  if (mode==READ)
    iofile_skip_bytes(4);
  else
    {
     iofile_put_byte(254);
     iofile_put_byte(3);
     iofile_put_byte(80); // P
     iofile_put_byte(75); // K :-)
    };

  for(t=1;t<36;t++)
  {
    switch(t)
      {
        case 9:  nbbytes+=io.tellg();
                 close_zipcode_file();
                 open_zipcode_file(filename.string(),'2',mode);
                 if (mode==READ)
                   iofile_skip_bytes(2);
                 else
		   {
		     iofile_put_byte(254);
		     iofile_put_byte(3);
		   };
                 break;
        case 17: nbbytes+=io.tellg();
                 close_zipcode_file();
                 open_zipcode_file(filename.string(),'3',mode);
                 if (mode==READ)
                   iofile_skip_bytes(2);
                 else
                   {
		     iofile_put_byte(254);
		     iofile_put_byte(3);
                   }
                 break;
        case 26: nbbytes+=io.tellg();
                 close_zipcode_file();
                 open_zipcode_file(filename.string(),'4',mode);
                 if (mode==READ)
                   iofile_skip_bytes(2);
                 else
                   {
		     iofile_put_byte(254);
		     iofile_put_byte(3);
                   }
                 break;
      }

    if (option_more_verbose) cout << "Track " << setw(2) << t << ": ";

    for(s=0;s<nbsectors_of_track(t);s++)
      {
       if (mode==READ)
	 {
	   err=zipcode_read_sectordata(t);
	   if (err) return err;
	 }
       else
          zipcode_write_sectordata(t,interleave_sector(t,s));

       nbzipsectors++;
      }

  if (option_more_verbose) cout << endl;
  }

  nbbytes+=io.tellg();
  close_zipcode_file();
  if (mode==READ)
    {
     if (nbzipsectors!=nbsectors)
       cerr << "Warning: Archive doesn't contain full disk." << endl;
     else
       if (option_verbose)
         cout << "Successfully read zipcode archive ("
              << nbbytes
              << " bytes / ";
    }
  else if (option_verbose)
         cout << "Successfully wrote zipcode archive ("
              << nbbytes
              << " bytes / ";

  if (option_verbose)
    cout << 100*nbbytes/174848 << "%)." << endl;

  return 0;
}

int c1541disk::zipcode_read_sectordata(int trk)
{
 int t, s, bit6set, bit7set, err;

 t=iofile_get_byte();
 s=iofile_get_byte();

 bit6set=((t& 64)== 64);
 bit7set=((t&128)==128);

 if (bit6set && bit7set)
   return 1;

 t = t & 63;

 if (t!=trk || t>35 || t<0)
   return 2;

 if (bit7set) {
               if (option_more_verbose)
                 cout << "R";
	       err=zipdecode_rle_method(t,s);
	       if (err) return err;
              }
 else if(bit6set) {
                   if (option_more_verbose)
                     cout << "F";
                   zipdecode_fill_method(t,s);
                  }
 else {
       if (option_more_verbose)
         cout << "W";
       zipdecode_write_through_method(t,s);
      }
 return 0;
}

void c1541disk::zipdecode_write_through_method(int trk, int sec)
{
 int i,c;

 for(i=0;i<256;i++)
   {
     c=iofile_get_byte();
     put_byte(trk,sec,i,c);
   };
}

void c1541disk::zipdecode_fill_method(int trk, int sec)
{
 int i,c;

 c=iofile_get_byte();
 for(i=0;i<256;i++) put_byte(trk,sec,i,c);
}

int c1541disk::zipdecode_rle_method(int trk, int sec)
{
 int i,j;
 int repeatmarker, nbsubblocks, count, value;
 int byte=0;

 nbsubblocks =iofile_get_byte();
 repeatmarker=iofile_get_byte();

 for(i=0;i<nbsubblocks;i++)
 {
   value=iofile_get_byte();
   if (value!=repeatmarker)
     {
      put_byte(trk, sec, byte, value);
      byte++;
     }
   else
     {
      i+=2;
      count=iofile_get_byte();
      value=iofile_get_byte();

      for(j=0;j<count;j++,byte++)
         put_byte(trk, sec, byte, value);
     }
 }
 if (byte!=256)
   return 4;

 return 0;
}

void c1541disk::zipcode_error(char* comment)
{
 cerr << "Error: Zipcode archive corrupt! (" << comment << ")" << endl;
 exit(1);
}

void c1541disk::zipcode_write_sectordata(int trk, int sec)
{
  int v,
      i,
      data_bytes;

  v=get_byte(trk,sec,0);
  for(i=1;i<256 && v==get_byte(trk,sec,i);i++);
  if (i==256) {
               if (option_more_verbose)
                 cout << "F";
               zipwrite_fill_method(trk,sec,v);
              }
  else
    {
     data_bytes=zipwrite_rle_method(trk,sec,1,0);
     if (data_bytes<0)
       {
        if (option_more_verbose)
          cout << "W";
        zipwrite_write_through_method(trk,sec);
       }
     else
       {
        if (option_more_verbose)
	  cout << "R";
        zipwrite_rle_method(trk,sec,0,data_bytes);
       }
    }
}

void c1541disk::zipwrite_fill_method(int trk, int sec, int value)
{
 iofile_put_byte(trk&63|64);
 iofile_put_byte(sec);
 iofile_put_byte(value);
}

void c1541disk::zipwrite_write_through_method(int trk, int sec)
{
 int i;

 iofile_put_byte(trk&63);
 iofile_put_byte(sec);
 for(i=0;i<256;i++)
   iofile_put_byte(get_byte(trk,sec,i));
}

int c1541disk::zipcode_find_marker(int trk, int sec)
{
 int i,
     store[256],
     v;

 for(i=0;i<256;i++) store[i]=0;

 for(i=0;i<256;i++)
   {
    v=get_byte(trk, sec, i);
    store[v]=1;
   }

 for(i=0;i<256 && store[i]==1;i++);
 if (i==256) i=-1;
 return i;
}

int c1541disk::zipwrite_rle_method(int trk, int sec, int test, int data_bytes)
{
 int i, j,
     marker,
     repcounter,
     v, v_old,
     nbzipbytes=0;

 marker=zipcode_find_marker(trk, sec);
 if (marker<0) internal_error(2);

 if(!test)
    {
     iofile_put_byte(trk&63|128);
     iofile_put_byte(sec);
     iofile_put_byte(data_bytes);
     iofile_put_byte(marker);
    }

 repcounter=1;
 v_old=get_byte(trk,sec,0);
 for(i=1;i<256;i++)
 {
   v=get_byte(trk,sec,i);
   if(v==v_old) repcounter++;
   else {
         if (repcounter>3)
           {
            nbzipbytes+=3;
            if (!test)
	      {
		iofile_put_byte(marker);
		iofile_put_byte(repcounter);
		iofile_put_byte(v_old);
	      }
           }
         else
           {
            nbzipbytes+=repcounter;
            if (!test)
              for(j=0;j<repcounter;j++)
                iofile_put_byte(v_old);
           }
         v_old=v;
         repcounter=1;
        }
 };
 if (repcounter>3)
   {
    nbzipbytes+=3;
    if (!test)
      {
	iofile_put_byte(marker);
	iofile_put_byte(repcounter);
	iofile_put_byte(v_old);
      }
   }
 else
   {
    nbzipbytes+=repcounter;
    if (!test)
      for(j=0;j<repcounter;j++)
	iofile_put_byte(v_old);
   };

 if (nbzipbytes<256-2)
   return nbzipbytes;
 else return -1;
}
